; -------------------------------------------------------------------------------------------------------------- ;
; Beach Head - LOCODE.prg
; -------------------------------------------------------------------------------------------------------------- ;
; Memory Map
; -------------------------------------------------------------------------------------------------------------- ;
; $0000 - $00ff:  Zero Page Values
; $0100 - $01ff:  Processor Stack
; $0200 - $02ff:  Work Values and C64 system
; $0300 - $03ff:  Work Values and C64 system
; $0400 - $07ff:  Screen: Status
; $0550 - $07a8:  Scores: Top Ten
; $0800 - $1dff:  Code:   Basic Program
; $1e00 - $3fff:  Code:   LOcode
; $4000 - $43ff:  Screen: 1st
; $4400 - $47ff:  Screen: 2nd - for scrolling tanks on the beach
; $4800 - $4fff:  Data:   Char sets
; $5000 - $6577:  Data:   Sprites
; $6850 - $bfff:  Code:   HIcode
; $c000 - $cfff:  <unused>
; $f000 - $feff:  <unused>
; $ff00 - $ffff:  Copy:   Kernel
; -------------------------------------------------------------------------------------------------------------- ;
; Testing
; -------------------------------------------------------------------------------------------------------------- ;
TANK_TEST               set 0                       ; 0=no 1=yes 2=yes and short strand
; -------------------------------------------------------------------------------------------------------------- ;
                        * equ BH_MemCodeLo          ; Beach Head - LOCODE.prg
; -------------------------------------------------------------------------------------------------------------- ;
                        incdir  ..\inc              ; C64 System Includes

C64CIA1                 include cia1.asm            ; Complex Interface Adapter (CIA) #1 Registers  $DC00-$DC0F
C64CIA2                 include cia2.asm            ; Complex Interface Adapter (CIA) #2 Registers  $DD00-$DD0F
C64SID                  include sid.asm             ; Sound Interface Device (SID) Registers        $D400-$D41C
C64VicII                include vic.asm             ; Video Interface Chip (VIC-II) Registers       $D000-$D02E
C64Kernel               include kernel.asm          ; Kernel Vectors
C64Colors               include color.asm           ; Colour RAM Address / Colours
C64Memory               include mem.asm             ; Memory Layout
C64StdMem               include std_mem.asm         ; Standard Zeropage / Stack / Vector

Game                    include inc\BH_Game.asm     ; Game Variables
ZeroPage                include inc\BH_Zpg.asm      ; Zero Page Addresses
; -------------------------------------------------------------------------------------------------------------- ;
; GameStart             Does    : Init all - MainLoop - Reset all - Return to BASIC
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GameStart               subroutine                  ; .hbu001. - BASIC 210
                        jsr TabSpritesWaitsInits    ; 
                        jsr C64Init                 ; 
                        jsr PlayerInit              ; 
                        
; -------------------------------------------------------------------------------------------------------------- ;
TankTesting             if TANK_TEST > 0            ; .test.
                        
                        lda #$04                    ; .test.
                        sta TabPlayerLives          ; .test.
                        sta TabPlayerLivesSav       ; .test.
                        lda #$08                    ; .test.
                        sta TabPlayerTanksSav       ; .test.
                        sta TabPlayerTanksLeft      ; .test.
                        
                        lda #TabBayMapSeaChaptersNo ; .test.
                        sta TabBayMapSeaChapters    ; .test.
                        
                        lda #>.PrepBasic            ; .test. - stack return address
                        pha                         ; .test. - for a good return point
                        lda #<.PrepBasic - $01      ; .test.
                        pha                         ; .test.
                        
                        lda #BH_BeachMinPosX        ; .test.
                        sta SP0X                    ; .test.
                        lda #BH_BeachMinPosY        ; .test.
                        sta SP0Y                    ; .test.
                        jsr Chapter_TanksOnBeach    ; .test.
                        
                        endif                       ; .test.
; -------------------------------------------------------------------------------------------------------------- ;
                        
.CallMainLoop           jsr Chapter_BayAreaMap      ; with main loop split into chapter loops
                        
.PrepBasic              jsr C64Reset                ; 
                        
                        lda #$00                    ; reset Timer B
                        sta CIACRB                  ; CIA1($DC0F) Control Register B
                        
GameStartX              rts                         ; return to BASIC
; -------------------------------------------------------------------------------------------------------------- ;
; C64Init               Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
C64Init                 subroutine                  ; 
                        ldx #$00                    ; 
.CopyKernalToRam        lda Kernel_Vectors,x        ; and IRQ routine
                        sta Kernel_Vectors,x        ; 
                        dex                         ; 
                        bne .CopyKernalToRam        ; greater/equal
                        
                        lda #$00                    ; 
                        sta CIACRA                  ; CIA1($DC0E) Control Register A
                        
                        lda #$7f                    ; .#######
                        sta CIAICR                  ; CIA1($DC0D) Interrupt Control
                        sta CI2ICR                  ; CIA2($DD0D) Interrupt Control Register
                        
                        lda #<IRQ_START             ; 
                        sta CINV_Lo                 ; 
                        lda #>IRQ_START             ; 
                        sta CINV_Hi                 ; 
                        
                        lda #TabIRQScreenPartStatus ; 
                        sta TabIRQScreenPartFlag    ; 
                        
                        jsr IRQ_SetScreenGfx        ; 
                        
                        sei                         ; 
                        jsr IRQ_ScrollSoft          ; 
                        
                        lda #VIC_IrqSetRaster       ; 
                        sta IRQMASK                 ; VIC($D01A) IRQ Mask Register
                        cli                         ; 
                        
                        lda #B_Koff                 ; 
                        sta R6510                   ; 
                        
C64InitX                rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; C64Reset              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
C64Reset                subroutine                  ; 
                        lda #$00                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        lda #BIKon                  ; basic/kernal on
                        sta R6510                   ; 
                        
                        sei                         ; 
                        lda #$00                    ; 
                        sta IRQMASK                 ; VIC($D01A) IRQ Mask Register
                        
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        and #VIC_MCM_Off            ; ###.#### - multi color mode
                        sta SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        
                        lda #[VIC_Video_0400 + VIC_CharS_1000 + VIC_CharS_unus] ; scrn=$0400-$07e7 chr=$1000-$17ff
                        sta VMCSB                   ; VIC($D018) VIC Chip Memory Control
                        
                        lda #$c4 + VIC_MemBank_0    ; ##...### - $03 = $0000-$3fff
                        sta CI2PRA                  ; CIA2($DD00) Data Port A - Bits 0-1 = VIC mem bank
                        
                        lda TabBackGroundColorStat  ; 
                        sta BGCOL0                  ; VIC($D021) Background Color 0
                        
                        lda #<CINV_Ini              ; restore IRQ
                        sta CINV_Lo                 ; 
                        
                        lda #>CINV_Ini              ; 
                        sta CINV_Hi                 ; 
                        
                        ldx #$1b                    ; 
                        ldy #$41                    ; 
                        stx TIMALO                  ; CIA1($DC04) Timer A (low byte)
                        sty TIMAHI                  ; CIA1($DC05) Timer A (high byte)
                        
                        lda #$81                    ; #......# - enable timer A interrupts
                        sta CIAICR                  ; CIA1($DC0D) Interrupt Control
                        
                        lda CIACRA                  ; CIA1($DC0E) Control Register A
                        and #$80                    ; #.......
                        ora #$11                    ; ...#...# - timer A start
                        sta CIACRA                  ; CIA1($DC0E) Control Register A
                        
                        cli                         ; 
C64ResetX               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TabSpritesWaitsInits  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TabSpritesWaitsInits    subroutine                  ; 
                        ldx #$07                    ; 
.InitLoop               lda #$00                    ; 
                        sta TabVicScrollSoftPosX    ; 
                        sta TabBulletHight,x        ; 
                        sta TabPlaneDirReLi,x       ; TabPlaneDirReLiStop
                        sta TabPlaneDirUpDo,x       ; TabPlaneDirUpDo_None
                        sta TabPlaneFlies,x         ; 
                        sta TabMoveSpriteLeRiTime,x ; 
                        sta TabMoveSpriteUpDoTime,x ; 
                        sta TabMoveSpriteLeRi,x     ; TabMoveSprite_Ri
                        sta TabMoveSpriteUpDo,x     ; TabMoveSprite_Up
                        sta TabPlaneSpeedAppr,x     ; TabPlaneSpeedApprNone
                        sta TabSpriteStatus,x       ; TabSpriteStatusFree
                        sta TabPlaneDistance,x      ; 
                        sta TabSpriteStepCount,x    ; 
                        sta TabWaitTankEneShoot     ; TabWaitTankEneShootNo
                        
                        lda #$01                    ; 
                        sta TabWaitTimes_Block01,x  ; IRQ dependent wait times
                        sta TabWaitTimes_Block03,x  ; 
                        sta TabWaitTimes_Block05,x  ; 
                        sta TabWaitFlags_Block01,x  ; 
                        sta TabWaitFlags_Block03,x  ; 
                        sta TabWaitFlags_Block05,x  ; 
                        sta TabWaitTimes_Block01Sav,x ; 
                        sta TabWaitTimes_Block03Sav,x ; 
                        sta TabWaitTimes_Block05Sav,x ; 
                        sta TabSeaVesselActive,x    ; TabSeaVesselActiveYes
                        sta TabUserAction           ; 
                        
                        dex                         ; 
                        bpl .InitLoop               ; 
                        
                        lda #$20                    ; 
                        sta TabPlaySfxVoc2_07       ; 
                        
TabSpritesWaitsInitsX   rts                         ; 
; ------------------------------------------------------------------------------------------------------------- ;
; LoadStrandData        Function: .hbu016.
;                       Parms   : 
;                       Returns : 
; ------------------------------------------------------------------------------------------------------------- ;
LoadStrandData          subroutine                  ; 
.BasicOn                lda #BIKon                  ; basic/i-o/kernal on
                        sta R6510                   ; 
                        
.SetFileParms           lda #$02                    ; logical file number
                        ldx #$08                    ; 
                        ldy #$00                    ; 
                        jsr SETLFS                  ; KERNEL - $FFBA = set logical file parameters
                        
.SetFileName            lda #[TabStrandFileNameX - TabStrandFileName] ; length
                        ldx #<TabStrandFileNam_Level; 
                        ldy #>TabStrandFileNam_Level; 
                        jsr SETNAM                  ; KERNEL - $FFBD = set filename parameters
                        
.LoadFile               lda #$00                    ; flag: $00=Load $01=Check
                        ldx #<BH_MemStrand          ; load address low
                        ldy #>BH_MemStrand          ; load address high
                        jsr LOAD                    ; KERNEL - $FFD5 = load from device
                        
.ChkSuccess             jsr READST                  ; KERNEL - $FFB7 = read I/O status word
                        cmp #$40                    ; EndOfData input file
                        beq .SavEndAdr              ; successfully loaded
                        
.ReInitFileName         lda #"0"                    ; restart at 1st level
                        sta TabStrandFileNam_1      ; 
                        bne .LoadFile               ; always
                        
.SavEndAdr              stx TabDiskFileEndAdrLo     ; save loaded files end address
                        sty TabDiskFileEndAdrHi     ; 

.CloseFile              lda #$02                    ; logical file number
                        jsr CLOSE                   ; close file
                        
.SetNextLevel           inc TabStrandFileNam_1      ; set next level file
                        
.BasicOff               lda #B_Koff                 ; basic/kernal off
                        sta R6510                   ; 
                        
.ChkShoreFileMax        ldx TabPlayerSkillLevel     ; $01-$04
                        dex                         ; 
                        lda SkillBeachLevelNo,x     ; max skill file name + $01
                        cmp TabStrandFileNam_1      ; 
                        beq LoadStrandDataX         ; 
                        
                        lda #$00                    ; 
                        ldx #$04                    ; 
.DelFootOfHill          sta TabColPtrLo_Screen_06,x ; show foot of hill only in last shore level
                        sta TabColPtrHi_Screen_06,x ; 
                        dex                         ; 
                        bpl .DelFootOfHill          ; 
                        
LoadStrandDataX         rts                         ; 
; ------------------------------------------------------------------------------------------------------------- ;
; LoadChapterData       Function: .hbu013.
;                       Parms   : 
;                       Returns : 
; ------------------------------------------------------------------------------------------------------------- ;
LoadChapterData         subroutine                  ; 
                        txa                         ; .hbu013.
                        pha                         ; .hbu013. - save level file no
                        
.InitSfx                jsr InitChapterSfx          ; .hbu013.
                        
.BasicOn                lda #BIKon                  ; basic/i-o/kernal on
                        sta R6510                   ; 
                        
.SetFileName            lda TabChapterFileLen,x     ; 
                        pha                         ; save len
                        ldy TabChapterFileAdrHi,x   ; 
                        lda TabChapterFileAdrLo,x   ; 
                        tax                         ; 
                        pla                         ; get len
                        jsr SETNAM                  ; KERNEL - $FFBD = set filename parameters
                        
.SetFileNumbers         lda #$02                    ; logical file number
                        ldx #$08                    ; device number                                    
                        ldy #$00                    ; secondary command number - $00 = new LOAD address in X/Y
                        jsr SETLFS                  ; KERNEL - $FFBA = set logical file parameters
                        
.LoadChapter            lda #$00                    ; flag: $00=Load $01=Check
                        ldx #<BH_MemScreenSetA      ; load address low
                        ldy #>BH_MemScreenSetA      ; load address high
                        jsr LOAD                    ; KERNEL - $FFD5 = load from device
                        
                        stx TabDiskFileEndAdrLo     ; save loaded files end address
                        sty TabDiskFileEndAdrHi     ; 
                        
                        lda #$02                    ; logical file number
                        jsr CLOSE                   ; close file
                        
.BasicOff               lda #B_Koff                 ; basic/kernal off
                        sta R6510                   ; 
                        
                        jsr MoveChapterColor        ; 
                        jsr MoveStatusScreen        ; .hbu013.
                        
                        pla                         ; .hbu013. - get level file no
                        beq .GetRaster              ; .hbu013. - no player/level display on start screen
                        
                        jsr ScreenStatusShowPlayer  ; .hbu013.
                        jsr ScreenStatusShowLevel   ; .hbu013.
                        
.GetRaster              lda Chapter_Screen_Values   ; .hbu013.
                        sta TabControlRaster        ; .hbu013.
                        
LoadChapterDataX        rts                         ; 
; ------------------------------------------------------------------------------------------------------------- ;
; ClearChapterColor     Function: 
;                       Parms   : 
;                       Returns : 
; ------------------------------------------------------------------------------------------------------------- ;
ClearChapterColor       subroutine                  ; 
                        lda #$00                    ; BLACK
                        tax                         ; byte count
                        ldy #$04                    ; page count
                        
.__ClrColorRamHi        equ * + $02                 ; 
.ClrColorRam            sta COLORAM,x               ; 
                        
                        inx                         ; byte count
                        bne .ClrColorRam            ; 
                        
.IncGetPage             inc .__ClrColorRamHi        ; to
                        
                        dey                         ; page count
                        bne .ClrColorRam            ; 
                        
.RestorePointer         lda #>COLORAM               ; 
                        sta .__ClrColorRamHi        ; 
                        
ClearChapterColorX      rts                         ; 
; ------------------------------------------------------------------------------------------------------------- ;
; MoveChapterColor      Function: 
;                       Parms   : 
;                       Returns : 
; ------------------------------------------------------------------------------------------------------------- ;
MoveChapterColor        subroutine                  ; 
                        ldx #$00                    ; byte count
                        ldy #$04                    ; page count
                        
.GetBaseColors          lda Colors_Screen_Base + $00; .hbu013.
                        sta TabBackGroundColor      ; .hbu013.
                        lda Colors_Screen_Base + $01; .hbu013.
                        sta BGCOL1                  ; .hbu013.
                        lda Colors_Screen_Base + $02; .hbu013.
                        sta BGCOL2                  ; .hbu013.
                        
.__GetChapterColorHi    equ * + $02                 ; 
.GetChapterColor        lda Colors_Screen,x;        .hbu013.
.__SetChapterColorHi    equ * + $02                 ; 
.SetChapterColor        sta COLORAM,x               ; 
                        
                        inx                         ; byte count
                        bne .GetChapterColor        ; 
                        
.IncGetPage             inc .__GetChapterColorHi    ; from 
                        inc .__SetChapterColorHi    ; to
                        
                        dey                         ; page count
                        bne .GetChapterColor        ; 
                        
.RestorePointer         lda #>BH_MemScreenSetB      ; 
                        sta .__GetChapterColorHi    ; 
                        lda #>COLORAM               ; 
                        sta .__SetChapterColorHi    ; 
                        
MoveChapterColorX       rts                         ; 
; ------------------------------------------------------------------------------------------------------------- ;
; MoveStatusScreen      Function: 
;                       Parms   : 
;                       Returns : 
; ------------------------------------------------------------------------------------------------------------- ;
MoveStatusScreen        subroutine                  ; 
                        ldx Chapter_Screen_Values + $01 ; copy status lenght
                        clc                         ; 
                        lda Chapter_Screen_Values + $02 ; copy status offset
                        adc #<Screen_Graphics       ; 
                        sta .__GetStatusScreenLo    ; 
                        lda Chapter_Screen_Values + $03 ; 
                        adc #>Screen_Graphics       ; 
                        sta .__GetStatusScreenHi    ; 
                        
                        clc                         ; 
                        lda Chapter_Screen_Values + $02 ; copy status offset
                        adc #<BH_MemScreenStatus    ; 
                        sta .__SetStatusScreenLo    ; 
                        lda Chapter_Screen_Values + $03 ; 
                        adc #>BH_MemScreenStatus    ; 
                        sta .__SetStatusScreenHi    ; 
                        
.__GetStatusScreenLo    equ * + $01                 ; 
.__GetStatusScreenHi    equ * + $02                 ; 
.GetStatusScreen        lda Screen_Graphics,x       ;      .hbu013.
.__SetStatusScreenLo    equ * + $01                 ; 
.__SetStatusScreenHi    equ * + $02                 ; 
.SetStatusScreen        sta BH_MemScreenStatus,x    ;      .hbu013.
                        dex                         ; 
                        cpx #$ff                    ; 
                        bne .GetStatusScreen        ; 
                        
MoveStatusScreenX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckGameIdleTime     Does    : Check if in game (reset TOD2 then) or demo mode
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckGameIdleTime       subroutine                  ; 
                        lda TabGameMode             ; 
                        beq .ChkUserAction          ; TabGameModeGame   
                        
                        rts                         ; TabGameModeDemo
                        
.ChkUserAction          lda TabUserAction           ; 
                        beq .ChkIdleTime            ; TabUserActionNo   
                        
.SetUserAction          lda #TabUserActionNo        ; 
                        sta TabUserAction           ; 
                        
                        jsr ResetTOD2               ; reset idle time
                        
.Exit                   rts                         ; 
                        
.ChkIdleTime            lda TO2MIN                  ; CIA2($DD0A) Time of Day Clock Minutes
                        cmp #$01                    ; up to 1 minute idle time is tolerable
                        bne CheckGameIdleTimeX      ; 
                        
.IdleTimeUp             lda #TabGameModeExit        ; idle too long - exit game
                        sta TabGameMode             ; 
                        
CheckGameIdleTimeX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckGamePause        Does    :           ; on C= commodore key
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckGamePause          subroutine                  ; 
                        lda CIAPRB                  ; CIA1($DC01) Data Port Register B
                        eor #$ff                    ; ########
                        and #CIA_KeyResCol_05       ; C= commodore key
                        bne .IniMerge               ; pressed
                        
                        rts                         ; 
                        
.IniMerge               ldy #$07                    ; 
                        lda #$00                    ; 
.MergeLoop              clc                         ; 
                        ldx TabSpriteActive,y       ; 
                        beq .MrgStoreSpritesActive  ; 
                        
                        sec                         ; 
                        
.MrgStoreSpritesActive  rol a                       ; merge store into one byte
                        
                        dey                         ; 
                        bpl .MergeLoop              ; 
                        
.SavStoreSpritesActive  tax                         ; save merged ac
                        
                        lda #$00                    ; 
.ClrStoreSpritesActive  jsr SpritesAcToTabActive    ; 
                        
                        lda #$00                    ; 
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
                        sta TO2HRS                  ; CIA2($DD0B) Time of Day Clock Hours
                        jsr CheckGamePauseDelay     ; 
                        
.PauseGame              lda CIAPRB                  ; CIA1($DC01) Data Port Register B
                        eor #$ff                    ; ########
                        and #CIA_KeyResCol_05       ; C= commodore key
                        beq .PauseGame              ; 
                        
                        jsr CheckGamePauseDelay     ; 
                        
                        txa                         ; restore merged ac
.RestoreStoreAC_9789    jsr SpritesAcToTabActive    ; 
                        
                        lda TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        sta TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        
                        lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        sta TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        
CheckGamePauseX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckGamePauseDelay   Does    : Wait time after GAME PAUSE to ready the controller again
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckGamePauseDelay     subroutine                  ; 
                        lda #$30                    ; 
                        sta TabWaitTimePause        ; 
                        
.IniWait                lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagPause        ; 
                        
.Delay                  lda TabWaitFlagPause        ; IRQ_Exec_Waits
                        bne .Delay                  ; 
                        
CheckGamePauseDelayX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ClearGfxScreen        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ClearGfxScreen          subroutine                  ; 
                        lda #$00                    ; 
                        ldx #$07                    ; 
.ClrSpritePtrs          sta BH_SpritePointers_Chr,x ; 
                        dex                         ; 
                        bpl .ClrSpritePtrs          ; 
                        
.SetScreen              lda #$00                    ; screen B
                        sta TabTankActiveScreen     ; 
                        
                        ldx #BH_Scrn_Gfx_Rows - $01 ; number of rows
.GetGfxScreenRowPtr     jsr TankObstIniScrnOutPtr   ; 
                        
                        ldy #BH_Scrn_RowLen - 1     ; 
                        lda #$ff                    ; 
.ClrGfxScreen           sta (BLZ_ScrnRowPtrGfxOut),y; 
                        dey                         ; 
                        bpl .ClrGfxScreen           ; 
                        
                        dex                         ; 
                        bpl .GetGfxScreenRowPtr     ; 
                        
ClearGfxScreenX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetRND                Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetRND                  subroutine                  ; 
                        lda #$01                    ; .......# - Start Timer B (1=start, 0=stop)
                        sta CIACRB                  ; CIA1($DC0F) Control Register B
                        
                        lda RASTER                  ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                        eor TIMBLO                  ; CIA1($DC06) Timer B (low byte)
                        eor TIMBHI                  ; CIA1($DC07) Timer B (high byte)
                        adc TabRndSeed              ; 
                        sta TabRndSeed              ; 
                        
GetRNDX                 rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetRND_4a_93          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetRND_4a_93            subroutine                  ; 
                        jsr GetRND                  ; 
                        
.ChkLimit               cmp #$4a                    ; 
                        bcs .Add_21                 ; greater/equal
                        
                        adc #$4a                    ; lower
                        rts                         ; 
                        
.Add_21                 adc #$20                    ; 
GetRND_4a_93X           jmp .ChkLimit               ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SwitchPlayerAndValues Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SwitchPlayerAndValues   subroutine                  ; 
                        ldx TabPlayerActive         ; 
                        jsr PlayerValuesSave        ; 
                        
                        lda TabPlayerInGame         ; 
                        cmp #$02                    ; 
                        beq .ChkPlayer_02           ; 
                        
.RestorePlayerValues    ldx TabPlayerActive         ; 
                        jsr PlayerValuesRestore     ; 
                        
                        lda #$00                    ; single player mode - player #1
                        rts                         ; 
                        
.ChkPlayer_02           cpx #$01                    ; switch players
                        bne .SetPlayer_01           ; 
                        
                        ldx #$00                    ; two player mode - player #1
                        stx TabPlayerActive         ; 
                        jmp .RestorePlayerValues    ; 
                        
.SetPlayer_01           ldx #$01                    ; two player mode - player #2
                        stx TabPlayerActive         ; 
                        jsr PlayerValuesRestore     ; 
                        
                        lda #$01                    ; player #2
SwitchPlayerAndValuesX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerSkillLevelInit  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerSkillLevelInit    subroutine                  ; 
                        ldx TabPlayerSkillLevel     ; 
                        dex                         ; 
                        
                        lda SkillCaveTorpLeRi_01,x  ; 
                        sta __CaveSkillTorpLeRi_01  ; 
                        
                        lda SkillCaveTorpUpDo_01,x  ; 
                        sta __CaveSkillTorpUpDo_01  ; 
                        
                        lda SkillCaveTorpLeRi_02,x  ; 
                        sta __CaveSkillTorpLeRi_02  ; 
                        
                        lda SkillCaveTorpUpDo_02,x  ; 
                        sta __CaveSkillTorpUpDo_02  ; 
                        
                        lda SkillCaveTorpLeRi_03,x  ; 
                        sta __CaveSkillTorpLeRi_03  ; 
                        
                        lda SkillCaveTorpUpDo_03,x  ; 
                        sta __CaveSkillTorpUpDo_03  ; 
                        
                        lda SkillCaveTorpLeRi_04,x  ; 
                        sta __CaveSkillTorpLeRi_04  ; 
                        
                        lda SkillCaveTorpLeRi_05,x  ; 
                        sta __CaveSkillTorpLeRi_05  ; 
                        
                        lda SkillCaveTorpLeRi_06,x  ; 
                        sta __CaveSkillTorpLeRi_06  ; 
                        
                        lda SkillCaveTorpLeRi_07,x  ; 
                        sta __CaveSkillTorpLeRi_07  ; 
                        
                        lda SkillSeaPlaneWaitAmmo,x ; 
                        sta __SeaPlaneSkillAmmoInc  ; 
                        
                        lda SkillSeaPlaneToHit,x    ; 
                        sta __SeaPlaneSkillToHit    ; 
                        
                        lda SkillSeaPlaneDmge,x     ; 
                        sta __SeaPlaneSkillDmge     ; 
                        
                        lda SkillSeaShipShots,x     ; 
                        sta __SeaShipSkillShots     ; 
                        
                        lda SkillBeachLevelNo,x     ; .hbu016.
                        sta __BeachLevelSkillNo     ; .hbu016.
                        
                        lda SkillBeachTankShoot,x   ; 
                        sta __BeachTankSkillShoot   ; 
                        
                        lda SkillHillCannonWait,x   ; 
                        sta __HillCannonSkillWait   ; 
                        
PlayerSkillLevelInitX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerValuesRestore   Does    : Sets game values for the actual player
;                       Expects : xr=player number
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerValuesRestore     subroutine                  ; 
                        lda TabPlayerLivesSav,x     ; 
                        sta TabPlayerLives          ; 
                        
                        lda TabPlayerCavePassSav,x  ; 
                        sta TabPlayerCavePassed     ; 
                        
                        lda TabPlayerTanksSav,x     ; 
                        sta TabPlayerTanksLeft      ; 
                        
                        lda TabPlayerShowScoresSav,x; 
                        sta TabPlayerShowScores     ; 
                        
                        lda TabPlayerScoreLoSav,x   ; 
                        sta TabPlayerScoreLo        ; 
                        
                        lda TabPlayerScoreHiSav,x   ; 
                        sta TabPlayerScoreHi        ; 
                        
                        lda TabPlayerHillTargsSav,x ; 
                        sta TabPlayerHillTargsToHit ; 
                        
                        lda TabPlayerBeachLvlSav,x  ; .hbu016.
                        sta TabBeachLevelNo         ; 
                        
                        lda TabPlayerVesselActSav,x ; 
                        sta TabPlayerVesselActive   ; 
                        
                        lda TabPlayerShipsSav,x     ; 
                        sta TabPlayerShips          ; 
                        
PlayerValuesRestoreX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerValuesSave      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerValuesSave        subroutine                  ; 
                        lda TabPlayerLives          ; 
                        sta TabPlayerLivesSav,x     ; 
                        
                        lda TabPlayerCavePassed     ; 
                        sta TabPlayerCavePassSav,x  ; 
                        
                        lda TabPlayerTanksLeft      ; 
                        sta TabPlayerTanksSav,x     ; 
                        
                        lda TabPlayerShowScores     ; 
                        sta TabPlayerShowScoresSav,x; 
                        
                        lda TabPlayerScoreLo        ; 
                        sta TabPlayerScoreLoSav,x   ; 
                        
                        lda TabPlayerScoreHi        ; 
                        sta TabPlayerScoreHiSav,x   ; 
                        
                        lda TabPlayerHillTargsToHit ; 
                        sta TabPlayerHillTargsSav,x ; 
                        
                        lda TabBeachLevelNo         ; .hbu016.
                        sta TabPlayerBeachLvlSav,x  ; 
                        
                        lda TabPlayerVesselActive   ; 
                        sta TabPlayerVesselActSav,x ; 
                        
                        lda TabPlayerShips          ; 
                        sta TabPlayerShipsSav,x     ; 
                        
PlayerValuesSaveX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerInit            Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerInit              subroutine                  ; 
                        ldx #$01                    ; 
.Players                lda #$0a                    ; 
                        sta TabPlayerLivesSav,x     ; 
                        
                        lda #$40                    ; 
                        sta TabMyFleetPosX          ; 
                        
                        lda #$01                    ; 
                        sta TabMyFleetMsbPosX       ; 
                        
                        lda #$32                    ; 
                        sta TabMyFleetPosY          ; 
                        
                        lda #$00                    ; 
                        sta TabJoysticAction        ; $00
                        sta SPSPCL                  ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                        sta SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        jsr SpritesAcToTabActive    ; 
                        
                        lda TabGameMode             ; 
                        bne .InitAll                ; TabGameModeDemo or TabGameModeExit
                        
                        lda #$00                    ; 
                        sta TO2HRS                  ; CIA2($DD0B) Time of Day Clock Hours
                        
.InitAll                lda #$00                    ; 
                        sta TabPlayerScoreLoSav,x   ; 
                        sta TabPlayerScoreHiSav,x   ; 
                        sta TabPlayerShipsSav,x     ; 
                        sta TabPlayerCavePassSav,x  ; 
                        sta TabPlayerTanksSav,x     ; 
                        sta TabPlayerShowScoresSav,x; 
                        
                        lda #$0a                    ; 
                        sta TabPlayerHillTargsSav,x ; 
                        
                        lda #TabShowScoresYes       ; 
                        sta TabPlayerShowScore,x    ; 
                        
                        lda #$3e                    ; ..#####. - one bit per ship
                        sta TabPlayerVesselActSav,x ; 
                        
                        lda #"0"                    ; .hbu016. - reinit level file name
                        sta TabStrandFileNam_10     ; .hbu016.
                        sta TabStrandFileNam_1      ; .hbu016.
                        
                        dex                         ; 
                        bpl .Players                ; 
                        
                        ldx #$00                    ; player 1
                        stx TabPlayerActive         ; 
                        
                        jsr PlayerValuesRestore     ; 
                        jsr PlayerSkillLevelInit    ; 
                        jsr ResetSid                ; 
                        
PlayerInitX             rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerAmmoInc         Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerAmmoInc           subroutine                  ; 
                        lda TabWaitFlagAmmoInc      ; 
                        beq .IncAmmo                ; TabWaitFlag_No
                        
                        rts                         ; 
                        
.IncAmmo                inc TabWaitFlagAmmoInc      ; TabWaitFlag_Yes
                        
                        lda TabPlayerAmmo           ; 
__SeaPlaneSkillAmmoInc  equ * + $01                 ; skill level adaptations
.ChkSkill               cmp #$0a                    ; 
                        bcs PlayerAmmoIncX          ; greater/equal
                        
                        inc TabPlayerAmmo           ; 
                        
PlayerAmmoIncX          rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerAmmoDec         Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerAmmoDec           subroutine                  ; 
                        lda TabPlayerAmmo           ; 
                        bne .DecAmmo                ; 
                        
                        rts                         ; 
                        
.DecAmmo                dec TabPlayerAmmo           ; 
                        
PlayerAmmoDecX          rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetTOD1             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetTOD1               subroutine                  ; 
                        lda #$00                    ; 
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
                        sta TODMIN                  ; CIA1($DC0A) Time of Day Clock Minutes
                        sta TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
                        sta TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        
ResetTOD1X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetTOD2             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetTOD2               subroutine                  ; 
                        lda #$00                    ; 
                        sta TO2HRS                  ; CIA2($DD0B) Time of Day Clock Hours
                        sta TO2MIN                  ; CIA2($DD0A) Time of Day Clock Minutes
                        sta TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        sta TO2TEN                  ; CIA2($DD08) Time of Day Clock Tenths
                        
ResetTOD2X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; WaitTOD1_Sec          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
WaitTOD1_Sec            subroutine                  ; 
                        jsr ResetTOD1               ; 
                        
.WaitTOD                lda TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
                        ldx TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        cmp TabWaitTimeTOD_Sec      ; 
                        bne .WaitTOD                ; 
                        
WaitTOD1X               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; WaitTOD1_10th         Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
WaitTOD1_10th           subroutine                  ; 
                        jsr ResetTOD1               ; 
                        
.WaitTOD                lda TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        cmp TabWaitTimeTOD_10th     ; 
                        bne .WaitTOD                ; 
                        
WaitTOD1_10thX          rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ClearDirInJoyAction   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ClearDirInJoyAction     subroutine                  ; 
                        eor #$ff                    ; flip selected bits
                        and TabJoysticAction        ; clear
                        sta TabJoysticAction        ; 
                        
ClearDirInJoyActionX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; AddScores             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
AddScore_100            subroutine                  ; 
                        lda #$01                    ; 
                        bne AddScores               ; always
                        
AddScore_200            subroutine                  ; 
                        lda #$02                    ; 
                        bne AddScores               ; always
                        
AddScore_400            subroutine                  ; 
                        lda #$04                    ; 
                        bne AddScores               ; always
                        
AddScore_600            subroutine                  ; 
                        lda #$06                    ; 
                        bne AddScores               ; always
                        
AddScore_800            subroutine                  ; 
                        lda #$08                    ; 
                        bne AddScores               ; always
                        
AddScore_1000           subroutine                  ; 
                        lda #$10                    ; 
                        bne AddScores               ; always
                        
AddScore_2000           subroutine                  ; 
                        lda #$20                    ; 
                        bne AddScores               ; always
                        
AddScore_9000           subroutine                  ; 
                        lda #$90                    ; 
                        
AddScores               subroutine                  ; 
.DecimalOn              sed                         ; 
                        
                        clc                         ; 
                        adc TabPlayerScoreLo        ; 
                        sta TabPlayerScoreLo        ; 
                        bcc .DecimalOff             ; 
                        
                        lda #$00                    ; 
                        adc TabPlayerScoreHi        ; 
                        sta TabPlayerScoreHi        ; 
                        
.DecimalOff             cld                         ; 
AddScoresX              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SubScores             Does    : Subtract 100 from actual score
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SubScores               subroutine                  ; .hbu020.
.DecimalOn              sed                         ; 
                        
                        lda TabPlayerScoreLo        ; 
                        sec                         ; 
                        sbc #$01                    ; 100
                        sta TabPlayerScoreLo        ; 
                        bcs .DecimalOff             ; 
                        
                        lda TabPlayerScoreHi        ; 
                        sbc #$00                    ; 
                        sta TabPlayerScoreHi        ; 
                        bcs .DecimalOff             ; 
                        
                        lda #$00                    ; set to min
                        sta TabPlayerScoreLo        ; 
                        sta TabPlayerScoreHi        ; 
                        
.DecimalOff             cld                         ; 
SubScoresX              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MakeDigit             Does    : 
;                       Expects : ac=digit
;                       Returns : xr=left nibble yr=right nibble
; -------------------------------------------------------------------------------------------------------------- ;
MakeDigit               subroutine                  ; 
                        pha                         ; save
                        and #$0f                    ; ....####
                        
                        clc                         ; 
                        adc #$30                    ; make digit
                        tay                         ; 
                        
                        pla                         ; restore
                        lsr a                       ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        
                        clc                         ; 
                        adc #$30                    ; make digit
                        tax                         ; 
                        
MakeDigitX              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MakeDecimal           Does    : 
;                       Expects : 
;                       Returns : xr=100 yr=10_1
; -------------------------------------------------------------------------------------------------------------- ;
MakeDecimal             subroutine                  ; 
                        ldy #$00                    ; 
.Sub100                 sta TabWrkMakeDecimal       ; 
                        sec                         ; 
                        sbc #$64                    ; 
                        bcc .Set100                 ; lower 100 - care for 10
                        
                        iny                         ; 
                        jmp .Sub100                 ; 
                        
.Set100                 tya                         ; 
                        tax                         ; count 100
                        lda TabWrkMakeDecimal       ; 
                        
                        ldy #$00                    ; 
.Sub10                  sta TabWrkMakeDecimal       ; 
                        sec                         ; 
                        sbc #$0a                    ; 
                        bcc .Set10_1                ; 
                        
                        iny                         ; 
                        jmp .Sub10                  ; 
                        
.Set10_1                tya                         ; 
                        asl a                       ; shift 10 to left halfbyte
                        asl a                       ; 
                        asl a                       ; 
                        asl a                       ; 
                        clc                         ; 
                        adc TabWrkMakeDecimal       ; add 1
                        tay                         ; 
                        
MakeDecimalX            rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; InitVoc1              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
InitVoc1                subroutine                  ; 
                        jsr ResetVoc1               ; 
                        
                        lda #$28                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$0c                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$81                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
                        lda #$fb                    ; #####.##
                        and TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
InitVoc1X               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; InitVoc3              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
InitVoc3                subroutine                  ; 
                        jsr ResetVoc3               ; 
                        
                        lda #$20                    ; 
                        sta FRELO3                  ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                        lda #$01                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$c8                    ; 
                        sta PWLO3                   ; SID($D410) Oscillator 3 Pulse Waveform Width (low byte)
                        lda #$09                    ; 
                        sta PWHI3                   ; SID($D411) Oscillator 3 Pulse Waveform Width (high nybble)
                        lda #$20                    ; 
                        sta SUREL3                  ; SID($D414) Oscillator 3 Sustain/Release
                        lda #$41                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
InitVoc3X               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetSid              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetSid                subroutine                  ; 
                        ldy #$17                    ; 
                        lda #$00                    ; 
.SetInit                sta SID,y                   ; SID($D400)
                        dey                         ; 
                        bpl .SetInit                ; 
                        
                        lda #$1f                    ; 
                        sta SIGVOL                  ; SID($D418) Volume/Filter Select
                        
ResetSidX               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetVoc1             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetVoc1               subroutine                  ; 
                        ldy #$06                    ; 
                        lda #$00                    ; 
.IniVoc1                sta FRELO1,y                ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        dey                         ; 
                        bpl .IniVoc1                ; 
                        
ResetVoc1X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetVoc2             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetVoc2               subroutine                  ; 
                        ldy #$06                    ; 
                        lda #$00                    ; 
.IniVoc2                sta FRELO2,y                ; SID($D407) Oscillator 2 Freq Ctrl (low byte)
                        dey                         ; 
                        bpl .IniVoc2                ; 
                        
ResetVoc2X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetVoc3             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetVoc3               subroutine                  ; 
                        ldy #$06                    ; 
                        lda #$00                    ; 
.ResetVoc3              sta FRELO3,y                ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                        dey                         ; 
                        bpl .ResetVoc3              ; 
                        
ResetVoc3X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SetVoc2_Cannon        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SetVoc2_Cannon          subroutine                  ; 
                        jsr ResetVoc2               ; 
                        
                        lda TabVoc2AttDec           ; 
                        sta ATDCY2                  ; SID($D40C) Oscillator 2 Attack/Decay
                        lda TabVoc2FreqHi           ; 
                        sta FREHI2                  ; SID($D408) Oscillator 2 Freq Ctrl (high byte)
                        lda #$81                    ; 
                        sta VCREG2                  ; SID($D40B) Oscillator 2 Control
                        
SetVoc2_CannonX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SetVoc3_Cannon        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SetVoc3_Cannon          subroutine                  ; 
                        jsr ResetVoc3               ; 
                        
                        lda #$34                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$0a                    ; 
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        lda #$81                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
SetVoc3_CannonX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ChapterStartBell      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ChapterStartBell        subroutine                  ; 
                        lda #LT_GREEN               ; .hbu015. - reset to green
                        sta COLORAM + BH_Scrn_Stat_Row_05 + BH_Scrn_Stat_Box + $00 ; .hbu015.
                        sta COLORAM + BH_Scrn_Stat_Row_05 + BH_Scrn_Stat_Box + $01 ; .hbu015.
                        
                        lda #$02                    ; 
                        jsr ResetChapterBell        ; 
                        
                        lda #$02                    ; ......#.
                        ora TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
                        lda #$01                    ; 
                        sta TabWaitTimeBells        ; 
                        
                        jsr ResetTOD1               ; 
                        
.GoRingBell             jsr RingChapterBell         ; 
                        
                        lda TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
                        cmp #$03                    ; 
                        bcc .GoRingBell             ; 
                        
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
                        
                        lda #$fd                    ; ######.#
                        and TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
                        jsr ResetChapterBell        ; 
ChapterStartBellX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; RingChapterBell       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
RingChapterBell         subroutine                  ; 
                        lda #$02                    ; ......#.
                        bit TabSfxToPlay            ; 
                        bne .ChkExit                ; 
                        
.Exit                   rts                         ; 
                        
.ChkExit                lda TabWaitFlagPing         ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagPing         ; TabWaitFlag_Yes
                        
                        lda #$1e                    ; 
                        sta TabWaitTimeBells        ; 
                        
                        lda #$00                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        lda #$0e                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$24                    ; 
                        sta FRELO3                  ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                        lda #$0c                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$6b                    ; 
                        sta FRELO1                  ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        lda #$30                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$15                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
RingChapterBellX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetChapterBell      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetChapterBell        subroutine                  ; 
                        ldx #$01                    ; 
                        lda #$00                    ; 
.Reset                  sta FRELO1,x                ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        sta FRELO2,x                ; SID($D407) Oscillator 2 Freq Ctrl (low byte)
                        sta FRELO3,x                ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                        dex                         ; 
                        bpl .Reset                  ; 
                        
ResetChapterBellX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_START             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_START               subroutine                  ; 
                        lda #$01                    ; 
                        sta VICIRQ                  ; VIC($D019) Interrupt Flags - Latched flags cleared if set to 1
                        
                        lda TabIRQScreenPartFlag    ; 
                        bne IRQ_ScreenPartGfx       ; 
                        
.WaitIni                ldx #$05                    ; 
.Wait                   dex                         ; 
                        bne .Wait                   ; 
                        
                        jsr IRQ_ScreenPartStatus    ; 
                        
                        lda #TabIRQScreenPartGfx    ; 
                        sta TabIRQScreenPartFlag    ; 
                        
                        cli                         ; 
                        
                        jsr IRQ_Exec_Waits          ; 
                        jsr IRQ_MoveSprites         ; 
                        jsr IRQ_CheckJoystick       ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_Return            Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_Return              subroutine                  ; 
                        pla                         ; 
                        tay                         ; 
                        pla                         ; 
                        tax                         ; 
                        pla                         ; 
                        
IRQX                    rti                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_ScreenPartGfx     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_ScreenPartGfx       subroutine                  ; 
                        jsr IRQ_SetScreenGfx        ; 
                        jsr IRQ_ScrollSoft          ; 
                        
                        lda #TabIRQScreenPartStatus ; 
                        sta TabIRQScreenPartFlag    ; 
                        
                        lda TabScreenTankColorFlag  ; 
                        beq IRQ_ScreenPartGfxX      ; TabScreenTankColorNo   
                        
                        jsr IRQ_ScreenTankColors    ; 
IRQ_ScreenPartGfxX      jmp IRQ_Return              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_MoveSprites       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_MoveSprites         subroutine                  ; 
                        ldx #$07                    ; 
.GetNextSprite          lda TabSpriteActive,x       ; 
                        bne .ChkMoveHoriz           ; TabSpriteActiveYes
                        
.SetNextSprite          dex                         ; TabSpriteActiveNo
                        bpl .GetNextSprite          ; 
                        
.Exit                   rts                         ; 
                        
.ChkMoveHoriz           lda TabMoveSpriteLeRi,x     ; 
                        and #$7f                    ; .#######
.ChkAmount              beq .ChkMoveVert            ; no amount found
                        
                        jsr IRQ_GetSpriteMoveStep   ; 
                        
                        clc                         ; 
                        adc TabSpriteWaitHori,x     ; 
                        sta TabSpriteWaitHori,x     ; 
                        
                        sec                         ; 
                        sbc #TabMoveSpritesStepTime ; 
                        bcc .GetSpriteOff           ; 
                        
                        sta TabSpriteWaitHori,x     ; 
                        inc TabMoveSpritesStep      ; 
                        
.GetSpriteOff           jsr SpritesGetVicOffFromXr  ; yr = xr * 2
                        lda TabMoveSpriteLeRi,x     ; 
                        bmi .MoveLeft               ; TabMoveSprite_Le
                        
.MoveRight              lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
                        adc TabMoveSpritesStep      ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcc .ChkMoveVert            ; 
                        
.SetSpriteInMask        jsr SetAcToXrTimesMult2     ; 
                        eor MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .ChkMoveVert            ; 
                        
.MoveLeft               lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sec                         ; 
                        sbc TabMoveSpritesStep      ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcc .SetSpriteInMask        ; 
                        
.ChkMoveVert            lda TabMoveSpriteUpDo,x     ; 
                        and #$7f                    ; .#######
                        beq .ApproachPlanes         ; no amount found
                        
                        jsr IRQ_GetSpriteMoveStep   ; 
                        
                        clc                         ; 
                        adc TabSpriteWaitVert,x     ; 
                        sta TabSpriteWaitVert,x     ; 
                        
                        sec                         ; 
                        sbc #TabMoveSpritesStepTime ; 
                        bcc .MoveUp                 ; 
                        
                        sta TabSpriteWaitVert,x     ; 
                        inc TabMoveSpritesStep      ; 
                        
.MoveUp                 jsr SpritesGetVicOffFromXr  ; yr = xr * 2
                        lda TabMoveSpriteUpDo,x     ; 
                        bmi .MoveDown               ; TabMoveSprite_Do
                        
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc TabMoveSpritesStep      ; 
                        bcs .SetUp                  ; 
                        
                        lda #$b8                    ; 
                        
.SetUp                  sta SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        jmp .ApproachPlanes         ; 
                        
.MoveDown               lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        clc                         ; 
                        adc TabMoveSpritesStep      ; 
                        cmp #$ba                    ; 
                        bcc .SetDown                ; lower
                        
                        lda #$00                    ; 
                        
.SetDown                sta SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)

.ApproachPlanes         lda TabPlaneSpeedAppr,x     ; 
                        and #$7f                    ; .#######
                        beq .GoSetNextSprite        ; TabPlaneSpeedApprNone
                        
                        jsr IRQ_GetSpriteMoveStep   ; 
                        
                        clc                         ; 
                        adc TabPlaneWaitAppr,x      ; 
                        sta TabPlaneWaitAppr,x      ; 
                        
                        sec                         ; 
                        sbc #TabMoveSpritesStepTime ; 
                        bcc .ChkSpeed               ; 
                        
                        sta TabPlaneWaitAppr,x      ; 
                        inc TabMoveSpritesStep      ; 
                        
.ChkSpeed               lda TabPlaneSpeedAppr,x     ; 
                        bpl .IncSpriteStepCount     ; 
                        
.DecSpriteStepCount     lda TabSpriteStepCount,x    ; 
                        sec                         ; 
                        sbc TabMoveSpritesStep      ; 
                        sta TabSpriteStepCount,x    ; 
                        
                        jmp .GoSetNextSprite        ; 
                        
.IncSpriteStepCount     lda TabSpriteStepCount,x    ; 
                        clc                         ; 
                        adc TabMoveSpritesStep      ; 
                        sta TabSpriteStepCount,x    ; 
                        
.GoSetNextSprite        jmp .SetNextSprite          ; 
                        
IRQ_MoveSpritesX        equ *                       ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_GetSpriteMoveStep Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_GetSpriteMoveStep   subroutine                  ; 
                        sec                         ; 
                        sbc #$3c                    ; 
                        bcc .StepNone               ; 
                        
.StepOne                ldy #$01                    ; 
                        sty TabMoveSpritesStep      ; 
                        
                        rts                         ; 
                        
.StepNone               clc                         ; restore old value 1st
                        adc #$3c                    ; 
                        
                        ldy #$00                    ; 
                        sty TabMoveSpritesStep      ; 
                        
IRQ_GetSpriteMoveStepX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_SetScreenGfx      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_SetScreenGfx        subroutine                  ; 
                        lda #$c4 + VIC_MemBank_1    ; ##...##. - $02 = $4000-$7fff
                        sta CI2PRA                  ; CIA2($DD00) Data Port A - Bits 0-1 = VIC mem bank
                        
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        ora #VIC_MCM_On             ; ...#.... - Multi Color Mode=on
                        sta SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        
                        lda TabVicMemControl        ; 
                        sta VMCSB                   ; VIC($D018) VIC Chip Memory Control
                        
                        lda TabBackGroundColor      ; 
                        sta BGCOL0                  ; VIC($D021) Background Color 0
                        
IRQ_SetScreenGfxX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_ScrollSoft        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_ScrollSoft          subroutine                  ; 
                        lda SCROLY                  ; VIC($D011) VIC Control Register 1 (Vert Fine Scroll)
                        and #[VIC_SoftY_Clear & VIC_RasterHi_Off] ; $78 - .####... (isolate soft scroll bits)
                        ora TabVicScrollSoftPosY    ; 
                        sta SCROLY                  ; VIC($D011) VIC Control Register 1 (Vert Fine Scroll)
                        
                        lda TabControlRaster        ; .hbu013.
.StatusScreenSet        sta RASTER                  ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                        
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        and #VIC_SoftX_Clear        ; $f8 - #####... (isolate soft scroll bits)
                        ora TabVicScrollSoftPosX    ; 
                        sta SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        
                        lda #TabScrnTankBayScrollFinYes ; 
                        sta TabScrnTankBayScrollFin ; 
                        
IRQ_ScrollSoftX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_ScreenTankColors  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_ScreenTankColors    subroutine                  ; 
                        cli                         ; 
                        
                        ldx #$04                    ; 
                        ldy #$e0                    ; 
.__GetColorHi           equ * + $02                 ; 
.GetColor               lda BH_Scrn_Gfx_Save - $e0,y ; 
.__SetColorHi           equ * + $02                 ; 
                        sta COLORAM - $e0,y         ; 
                        iny                         ; 
                        bne .GetColor               ; 
                        
                        inc .__GetColorHi           ; 
                        inc .__SetColorHi           ; 
                        
                        dex                         ; 
                        bne .GetColor               ; 
                        
                        lda #>[BH_Scrn_Gfx_Save - $e0] ; 
                        sta .__GetColorHi           ; 
                        lda #>[COLORAM - $e0]       ; 
                        sta .__SetColorHi           ; 
                        
                        lda #TabScreenTankColorNo   ; 
                        sta TabScreenTankColorFlag  ; 
                        
IRQ_ScreenTankColorsX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_CheckJoystick     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_CheckJoystick       subroutine                  ; 
                        lda CIAPRA                  ; Data Port Register A
                        eor #$ff                    ; flip joystick action - $00 if set to $01
                        and #$1f                    ; ...##### - mask out joystick action
                        beq IRQ_CheckJoystickX      ; was none
                        
                        ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
                        lda #TabGameModeGame        ; interrupt demo on user action
                        sta TabGameMode             ; 
                        
                        lda #TabUserActionYes       ; 
                        sta TabUserAction           ; 
                        
IRQ_CheckJoystickX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_Exec_Waits        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_Exec_Waits          subroutine                  ; 
                        ldy #$28                    ; 5 blocks a 8 bytes
                        ldx #$00                    ; 
.GetNextFlag            lda TabWaitFlags,x          ; 
                        beq .SetNext                ; TabWaitMove_Yes
                        
                        dec TabWaitTimes,x          ; 
                        bne .SetNext                ; 
                        
                        lda #$00                    ; 
.ResetFlag              sta TabWaitFlags,x          ; TabWaitMove_Yes (has to be TabWaitTimes + $28)
                        
.RestoreTime            lda TabWaitTimesSav,x       ; 
                        sta TabWaitTimes,x          ; 
                        
.SetNext                inx                         ; 
                        dey                         ; 
                        bne .GetNextFlag            ; 
                        
IRQ_Exec_WaitsX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_ScreenPartStatus  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_ScreenPartStatus    subroutine                  ; 
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        and #[VIC_SoftX_Clear & VIC_MCM_Off | VIC_Cols40] ; $e8 - ###.#...
                        ora #$00                    ; 
                        tax                         ; 
                        
                        lda SCROLY                  ; VIC($D011) VIC Control Register 1 (Vert Fine Scroll)
                        and #[VIC_SoftY_Clear & VIC_RasterHi_Off] ; $78 - .####... (isolate soft scroll bits)
                        ora #$03                    ; ......##
                        tay                         ; 
                        
                        lda TabBackGroundColorStat  ; 
                        jsr IRQ_Wait                ; 
                        
                        stx SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        sta BGCOL0                  ; VIC($D021) Background Color 0
                        
                        lda #[VIC_Video_0400 + VIC_CharS_1000 + VIC_CharS_unus] ; scrn=$0400-$07e7 chr=$1000-$17ff
                        sta VMCSB                   ; VIC($D018) VIC Chip Memory Control
                        
                        lda #$c4 + VIC_MemBank_0    ; ##...### - $03 = $0000-$3fff
                        sta CI2PRA                  ; CIA2($DD00) Data Port A - Bits 0-1 = VIC mem bank
                        
                        lda #$ff                    ; 
                        sta RASTER                  ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                        
IRQ_ScreenPartStatusX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_Wait              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_Wait                subroutine                  ; 
                        pha                         ; 
                        txa                         ; 
                        
                        ldx #$21                    ; 
.Wait                   dex                         ; 
                        bne .Wait                   ; 
                        
                        tax                         ; 
                        pla                         ; 
                        
                        bit SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bit D6510                   ; 6510 On-Chip I/O Data Direction
                        
IRQ_WaitX               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpritesAcToTabActive  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpritesAcToTabActive    subroutine                  ; 
                        ldy #$07                    ; bit count
.ShiftNextBit           asl a                       ; shift a bit to carry
                        pha                         ; save ac
                        
                        lda #TabSpriteActiveNo      ; preset no
                        bcc .StoreBitInTab          ; 
                        
                        lda #TabSpriteActiveYes     ; 
                        
.StoreBitInTab          sta TabSpriteActive,y       ; 
                        
                        pla                         ; restore ac
                        dey                         ; bit count
                        bpl .ShiftNextBit           ; 
                        
SpritesAcToTabActiveX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpritesGetSprtCollReg Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpritesGetSprtCollReg   subroutine                  ; 
                        lda SPSPCL                  ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                        sta TabSprtSprtCollReg      ; 
                        
SpritesGetSprtCollRegX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpritesGetBackCollReg Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpritesGetBackCollReg   lda SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        sta TabSprtBackCollReg      ; 
                        
SpritesGetBackCollRegX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpritesEnableNoFromXr Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpritesEnableNoFromXr   subroutine                  ; 
                        jsr SetAcToXrTimesMult2     ; 
                        
                        ora SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
SpritesEnableNoFromXrX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpritesDisableNoFromXr Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpritesDisableNoFromXr  subroutine                  ; 
                        jsr SetAcToXrTimesMult2     ; 
                        
                        eor #$ff                    ; 
                        and SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
SpritesDisableNoFromXrX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpritesReset          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpritesReset            subroutine                  ; 
                        lda #$00                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPBGPR                  ; VIC($D01B) Sprite to Foreground Display Priority
                        sta TabJoysticAction        ; 
                        
                        lda SPSPCL                  ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                        lda SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        
SpritesResetX           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SetAcToXrTimesMult2   Does    : Shift ACC=$01 XR times left
;                       Expects : XR = count
;                       Returns : AC power of 2 - XR not changed
; -------------------------------------------------------------------------------------------------------------- ;
SetAcToXrTimesMult2     subroutine                  ; 
                        txa                         ; ac=xr
                        tay                         ; yr=xr=ac
                        
                        lda #$01                    ; 
                        
.NextPowerOfTwo         dey                         ; 
                        bmi SetAcToXrTimesMult2X    ; 
                        
                        asl a                       ; 
                        jmp .NextPowerOfTwo         ; 
                        
SetAcToXrTimesMult2X    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpritesGetVicOffFromXr Does   : yr = xr * 2
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpritesGetVicOffFromXr  subroutine                  ; 
                        txa                         ; 
                        asl a                       ; 
                        tay                         ; 
                        
SpritesGetVicOffFromXrX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckGameFinished     Does    : Returns to start screen if idle too long or dead or move left on JoyStick 1
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckGameFinished       subroutine                  ; 
                        lda TabGameMode             ; 
;                        bmi CheckGameFinishedX      ; TabGameModeExit
                        bmi GameOver_Rts_Two        ; TabGameModeExit - Game over
                        
                        lda CIAPRB                  ; CIA1($DC01) Data Port Register B
                        eor #$ff                    ; 
                        and #CIA_KeySelRow_06       ; .hbu003 . - "Q"-uit
                        bne .NoScores               ; 
                        
                        rts                         ; 
                        
.NoScores               lda #TabShowScoresDemo      ; no HiScore entry
                        sta TabPlayerShowScores     ; 
                        lda #TabGameModeGame        ; end demo mode
                        sta TabGameMode             ; 
                        
;CheckGameFinishedX      jmp GameOver_Rts_Two        ; back to BASIC via GameStart
; -------------------------------------------------------------------------------------------------------------- ;
; GameOver_Rts          Does    : Game over - Back to BASIC via GameStart
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GameOver_Rts_Two        subroutine                  ; 
                        pla                         ; level 2
                        pla                         ; 
                        
GameOver_Rts_One        subroutine                  ; 
                        pla                         ; level 1
                        pla                         ; 
                        
GameOver_RtsX           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ClearStatusScreen     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ClearStatusScreen       subroutine                  ; 
                        lda #VIC_Sprts_Off          ; 
                        sta SPENA                   ; sprites off
                        jsr SpritesAcToTabActive    ; 
                        
                        lda #$01                    ; .hbu013. - screen A and WHITE
                        sta TabTankActiveScreen     ; .hbu013.
                        sta TabStatusScrnTextColor  ; 
                        
                        lda #VIC_Sprts_Off          ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        sta TabBackGroundColor      ; BLACK
                        
                        jsr ResetSid                ; .hbu013.
                        jsr ClearGfxScreen          ; .hbu013.
                        jsr ClearChapterColor       ; .hbu013.
                        
ClearStatusScreenX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusGetReady  Does    : Display 'get ready' screen
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusGetReady    subroutine                  ; 
                        jsr ClearStatusScreen       ; 
                        
                        lda TabVicMemControlSetA    ; 
                        sta TabVicMemControl        ; SetA
                        
                        ldy #BH_Scrn_Ready_LeTxt    ; get ready
                        ldx #<TabTextStatusGetReady ; 
                        jsr ScreenStatusTextOutRow3 ; xr=input text offset  yr=output text offset
                        
                        lda TabPlayerActive         ; 
                        bne .GetPlayerTwo           ; 
                        
.GetPlayerOne           ldx #<TabTextStatusPlayerOne; 
                        jmp .OutPlayerNo            ; 
                        
.GetPlayerTwo           ldx #<TabTextStatusPlayerTwo; 
                        
.OutPlayerNo            ldy #BH_Scrn_Ready_RiTxt    ; 
                        jsr ScreenStatusTextOutRow3 ; xr=input text offset  yr=output text offset (Player two)
                        
.ShowTime               lda #$01                    ; .hbu013.
                        sta TabWaitTimeTOD_Sec      ; .hbu013.
                        jsr WaitTOD1_Sec            ; .hbu013.
                        
ScreenStatusGetReadyX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusGameOver  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusGameOver    subroutine                  ; 
                        jsr ClearStatusScreen       ; 
.GetTempScores          jsr ScreenStatusShowScore   ; displayed in wrong status row
                        
                        ldx #BH_Scrn_Stat_LenScore  ; 
.CpyTempScores          lda [BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_RiVal],x; 
                        sta [BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_GaOvr_RiVal],x; 
                        
                        lda #BLACK                  ; 
.ClrTempScores          sta [COLORAM + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_RiVal],x ; 
                        
                        lda #WHITE                  ; 
                        sta [COLORAM + BH_Scrn_Stat_Row_02 + BH_Scrn_GaOvr_RiVal],x ; 
                        
                        dex                         ; 
                        bpl .CpyTempScores          ; 
                        
                        lda #WHITE                  ; 
                        sta TabStatusScrnTextColor  ; 
                        
                        ldy #BH_Scrn_GaOvr_RiTxt    ; 
                        ldx #<TabTextStatusScore    ; 
                        jsr ScreenStatusTextOutRow2 ; xr=input text offset  yr=output text offset (Score)
                        
                        ldy #BH_Scrn_GaOvr_LeTxt    ; game over
                        ldx #<TabTextStatusGameOver ; 
                        jsr ScreenStatusTextOutRow4 ; xr=input text offset  yr=output text offset
                        
                        lda TabPlayerActive         ; 
                        bne .GetTxtPlayer_2         ; 
                        
                        ldx #<TabTextStatusPlayerOne; 
                        jmp .OutTxtPlayer_2         ; 
                        
.GetTxtPlayer_2         ldx #<TabTextStatusPlayerTwo; 
                        
.OutTxtPlayer_2         ldy #BH_Scrn_Ready_RiTxt    ; 
                        jsr ScreenStatusTextOutRow4 ; xr=input text offset  yr=output text offset
                        
.ShowTime               lda #$05                    ; 
                        sta TabWaitTimeTOD_Sec      ; 
                        jsr WaitTOD1_Sec            ; 
                        
ScreenStatusGameOverX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusTextOut   Does    : 
;                       Expects : xr=input text offset  yr=output text offset
;                                 xr=length for blank out if TabStatusScrnTextColor=$ff
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusTextOut     subroutine                  ; 
ScreenStatusTextOutRow1 lda #<BH_Scrn_Stat_Row_01   ; 
                        jmp .SetPtrLo               ; 
ScreenStatusTextOutRow2 lda #<BH_Scrn_Stat_Row_02   ; 
                        jmp .SetPtrLo               ; 
ScreenStatusTextOutRow3 lda #<BH_Scrn_Stat_Row_03   ; 
                        jmp .SetPtrLo               ; 
ScreenStatusTextOutRow4 lda #<BH_Scrn_Stat_Row_04   ; 
                        jmp .SetPtrLo               ; 
ScreenStatusTextOutRow5 lda #<BH_Scrn_Stat_Row_05   ; 
                        
.SetPtrLo               sta BLZ_ScrnRowPtrStatusOutLo ; 
                        sta BLZ_ScrnRowPtrColoRamOutLo ; 
                        
                        lda #>[BH_Scrn_Stat + BH_Scrn_Stat_Row_01] ; 
                        sta BLZ_ScrnRowPtrStatusOutHi ; 
                        
                        lda #>[COLORAM      + BH_Scrn_Stat_Row_01] ; 
                        sta BLZ_ScrnRowPtrColoRamOutHi ; 
                        
                        lda TabStatusScrnTextColor  ; 
                        bmi .GetBlank               ; flag: $ff - blanks out
                        
.GetText                lda TabTextStatus,x         ; 
                        beq ScreenStatusTextOutX    ; end of text
                        sta (BLZ_ScrnRowPtrStatusOut),y ; 
                        
                        lda TabStatusScrnTextColor  ; 
                        sta (BLZ_ScrnRowPtrColoRamOut),y ; 
                        
                        inx                         ; 
                        iny                         ; 
                        jmp .GetText                ; 
                        
.GetBlank               lda #$20                    ; 
.PutBlank               sta (BLZ_ScrnRowPtrStatusOut),y ; 
                        
                        iny                         ; 
                        dex                         ; 
                        bne .PutBlank               ; 
                        
ScreenStatusTextOutX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusTextLong  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusTextLong    subroutine                  ; 
                        cmp #$05                    ; .hbu022.
                        bcc .ShowDist               ; .hbu022.
                        
                        jmp ScreenStatusTextMiss    ; .hbu022. - do not show the exact miss distance
                        
.ShowDist               jsr ScreenStatusShowDist    ; 
                        
                        lda #BH_Scrn_Stat_ColorShotLong ; 
                        sta TabStatusScrnTextColor  ; 
                        sta COLORAM + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $05 ; 
                        
                        ldy #BH_Scrn_Stat_RiTxt     ; 
                        ldx #<TabTextStatusMLong    ; 
                        jsr ScreenStatusTextOutRow2 ; .hbu011. - xr=input text offset  yr=output text offset
                        
ScreenStatusTextLongX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusTextShort Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusTextShort   subroutine                  ; 
                        cmp #$05                    ; .hbu022.
                        bcc .ShowDist               ; .hbu022.
                        
                        jmp ScreenStatusTextMiss    ; .hbu022. - do not show the exact miss distance
                        
.ShowDist               jsr ScreenStatusShowDist    ; 
                        
                        lda #BH_Scrn_Stat_ColorShotShort ; 
                        sta TabStatusScrnTextColor  ; 
                        sta COLORAM + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $05 ; 
                        
                        ldy #BH_Scrn_Stat_RiTxt     ; 
                        ldx #<TabTextStatusMShort   ; 
                        jsr ScreenStatusTextOutRow2 ; .hbu011. - xr=input text offset  yr=output text offset
                        
ScreenStatusTextShortX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusTextDist  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusTextDist    subroutine                  ; 
                        lda #" "                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $00 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $01 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $02 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $03 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $05 ; 
                        
                        lda #BH_Scrn_Stat_ColorShotMiss ; 
                        sta TabStatusScrnTextColor  ; 
                        sta COLORAM + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $05 ; 
                        
                        ldy #BH_Scrn_Stat_RiTxt     ; 
                        ldx #<TabTextStatusMDist    ; 
                        jsr ScreenStatusTextOutRow2 ; .hbu011. - xr=input text offset  yr=output text offset
                        
ScreenStatusTextDistX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusTextMiss  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusTextMiss    subroutine                  ; 
                        lda #"?"                    ; "?"
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $00 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $01 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $02 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $03 ; 
                        
                        lda #$0d                    ; .hbu011. - "M"
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $05 ; 
                        
                        lda #BH_Scrn_Stat_ColorShotMiss ; 
                        sta TabStatusScrnTextColor  ; 
                        sta COLORAM + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $05 ; 
                        
                        ldy #BH_Scrn_Stat_RiTxt     ; 
                        ldx #<TabTextStatusMMiss    ; 
                        jsr ScreenStatusTextOutRow2 ; .hbu011. - xr=input text offset  yr=output text offset
                        
ScreenStatusTextMissX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowPlayer Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowPlayer  subroutine                  ; 
                        ldx TabPlayerActive         ; 
                        inx                         ; 
                        txa                         ; 
                        jsr MakeDigit               ; xr=left nibble yr=right nibble
                        sty BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_LeVal ; 
                        
ScreenStatusShowPlayerX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowLevel Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowLevel   subroutine                  ; 
                        lda TabPlayerSkillLevel     ; 
                        jsr MakeDigit               ; xr=left nibble yr=right nibble
                        
                        sty BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_LeVal ; 
                        
ScreenStatusShowLevelX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusDestroy   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusDestroy     subroutine                  ; 
                        lda TabSeaShipMsgDestroy    ; 
                        cmp #TabSeaShipMsgDestroyYes; 
                        beq .MsgOut                 ; 
                        
                        rts                         ; 
                        
.MsgOut                 lda TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
.FlickerTime            cmp #$03                    ; .hbu011.
                        beq .MsgClear               ; 
                        
.Flicker                lda TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        sta TabBackGroundColor      ; 
                        sta TabStatusScrnTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_LeTxt - $03 ; 
                        ldx #<TabTextStatusShipSunk ; 
                        jsr ScreenStatusTextOutRow2 ; .hbu011. - xr=input text offset  yr=output text offset
                        rts                         ; 
                        
.MsgClear               lda #$00                    ; 
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
                        sta TabSeaShipMsgDestroy    ; TabSeaShipMsgDestroyNo
                        
.RestoreSky             lda #CYAN                   ; 
                        sta TabBackGroundColor      ; 
                        
                        lda #$ff                    ; flag: blank
                        sta TabStatusScrnTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_LeTxt - $03 ; 
                        ldx #TabLenStatusShipSunk   ; 
                        jsr ScreenStatusTextOutRow2 ; .hbu011. - xr=length blank out  yr=output text offset
ScreenStatusDestroyX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowAmmo  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowAmmo    subroutine                  ; 
                        ldx #$00                    ; 
                        ldy TabPlayerAmmo           ; 
                        dey                         ; 
                        bmi .BlankOut               ; 
                        
.BoxesOut               lda #$5a                    ; display ammo boxes
                        sta [BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal],x ; 
                        
                        inx                         ; 
                        dey                         ; 
                        bpl .BoxesOut               ; 
                        
.BlankOut               cpx #BH_Scrn_Stat_LenAmmo   ; 
                        bcs ScreenStatusShowAmmoX   ; 
                        
                        lda #$20                    ; 
                        sta [BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal],x ; 
                        inx                         ; 
                        jmp .BlankOut               ; 
                        
ScreenStatusShowAmmoX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowShips Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowShips   subroutine                  ; 
                        ldx #$00                    ; 
                        ldy TabPlayerLives          ; 
                        beq .ChkMax                 ; 
                        
.GetIndicator           lda #$51                    ; "O"
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_RiVal,x ; indicators
                        
                        lda #BH_Scrn_Stat_ColorCavePass ; preset
                        
                        cpy TabPlayerShips          ; 
                        bcc .SetColor               ; lower
                        beq .SetColor               ; equal
                        
                        lda #BH_Scrn_Stat_ColorShips ; greater
                        
.SetColor               sta COLORAM + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_RiVal,x ; 
                        
                        inx                         ; 
                        dey                         ; 
                        bne .GetIndicator           ; 
                        
.ChkMax                 cpx #BH_ShipsMax            ; 
                        bne .BlankOut               ; 
                        
.Exit                   rts                         ; 
                        
.BlankOut               lda #$20                    ; " "
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_RiVal,x ; 
                        
                        inx                         ; 
ScreenStatusShowShipsX  jmp .ChkMax                 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowTanks Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowTanks   subroutine                  ; 
                        ldx #$00                    ; 
                        ldy TabPlayerTanksLeft      ; 
                        beq .ChkMax                 ; 
                        
.GetIndicator           lda #$57                    ; .hbu011.
                        sta [BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_RiVal],x ; 
                        
                        lda #BH_Scrn_Stat_ColorTanks ; 
.SetColor               sta [COLORAM + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_RiVal],x ; 
                        
                        inx                         ; 
                        dey                         ; 
                        bne .GetIndicator           ; 
                        
.ChkMax                 cpx #BH_TanksMax            ; .hbu011.
                        bne .BlankOut               ; 
                        
.Exit                   rts                         ; 
                        
.BlankOut               lda #$20                    ; " "
                        sta [BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_RiVal],x ; 
                        
                        inx                         ; 
ScreenStatusShowTanksX  jmp .ChkMax                 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowScore Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowScore   subroutine                  ; 
                        lda TabPlayerScoreHi        ; 
                        jsr MakeDigit               ; xr=left nibble yr=right nibble
.SetScores              stx BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_RiVal + $00 ; 
                        sty BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_RiVal + $01 ; 
                        
                        lda TabPlayerScoreLo        ; 
                        jsr MakeDigit               ; xr=left nibble yr=right nibble
                        stx BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_RiVal + $02 ; 
                        sty BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_RiVal + $03 ; 
                        
                        lda #$30                    ; "0"
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_RiVal + $04 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_RiVal + $05 ; 
                        
ScreenStatusShowScoreX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowDist  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowDist    subroutine                  ; 
                        jsr MakeDecimal             ; xr=100 yr=10_1
                        jsr MakeDigit               ; xr=left nibble yr=right nibble
                        stx BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $00 ; 
                        sty BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $01 ; 
                        
                        lda #$30                    ; "0"
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $02 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $03 ; 
                        
                        lda #$0d                    ; .hbu011. - "M"
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_RiVal + $05 ; 
                        
ScreenStatusShowDistX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowDegree Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowDegree  subroutine                  ; 
                        lda #$b3                    ; 
                        sec                         ; 
                        sbc TabCannonShotDegree     ; 
                        clc                         ; 
                        lsr a                       ; rightmost bit contains 10th
                        pha                         ; save degrees
                        bcc .GetZero10th            ; 
                        
.GetFive10th            lda #$05                    ; 
                        jmp .Out10th                ; 
                        
.GetZero10th            lda #$00                    ; 
                        
.Out10th                jsr MakeDigit               ; xr=left nibble yr=right nibble
                        sty BH_Scrn_Stat + BH_Scrn_Stat_Row_01 + BH_Scrn_Stat_RiVal + $03
                        
                        pla                         ; restore degrees
                        jsr MakeDecimal             ; xr=100 yr=10_1
                        
                        tya                         ; 
                        jsr MakeDigit               ; xr=left nibble yr=right nibble
                        stx BH_Scrn_Stat + BH_Scrn_Stat_Row_01 + BH_Scrn_Stat_RiVal + $00
                        sty BH_Scrn_Stat + BH_Scrn_Stat_Row_01 + BH_Scrn_Stat_RiVal + $01
                        
                        lda #$2e                    ; "."
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_01 + BH_Scrn_Stat_RiVal + $02
                        
ScreenStatusShowDegreeX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
GraphicsTables          include asm\Tables.asm      ; Gfx Tables / Beach Colums Info
; -------------------------------------------------------------------------------------------------------------- ;
Seg_INTRO               seg.u   INTRO               ; 
                        include asm\Cha00.Intro.asm ; 
Seg_MAP                 seg.u   MAP                 ; 
                        include asm\Cha01.BayMap.asm ; 
Seg_CAVE                seg.u   CAVE                ; 
                        include asm\Cha02.Cave.asm  ; 
Seg_PLANE               seg.u   PLANE               ; 
                        include asm\Cha03.SeaPlane.asm ; 
Seg_SHIP                seg.u   SHIP                ; 
                        include asm\Cha04.SeaShips.asm ; 
Seg_BEACH               seg.u   BEACH               ; 
                        include asm\Cha05.BayShore.asm ; 
Seg_HILL                seg.u   HILL                ; 
                        include asm\Cha06.Hill.asm  ; 
; -------------------------------------------------------------------------------------------------------------- ;
